/** @file   spawnevent.cpp
 * @brief   Implementation of SpawnEvent and SpawnItem - classes.
 * @version $Revision: 1.7 $
 * @author  Tomi Lamminsaari
 */

#include "spawnevent.h" // class's header file
#include <sstream>
#include "electricfence.h"
#include "teleporter.h"
#include "AnimId.h"
#include "warglobals.h"
#include "soundsamples.h"
#include "www_map.h"
#include "animplayer.h"
#include "soundsource.h"
#include "aicontroller.h"
#include "settings.h"
#include "www_assert.h"
#include "SpawnParameters.h"
#include "SpawnItems.h"

using std::string;
using std::istream;
using std::istringstream;
using namespace eng2d;

namespace WeWantWar {
  
SpawnItem::SpawnItem()
{
}

SpawnItem::~SpawnItem()
{
}

int SpawnItem::Read( istream& aIn )
{
  while ( true ) {
    if ( aIn.eof() == true ) {
      alert( "EOF while parsing ", this->GetLiteral(EOpeningTag).c_str(), "element",
             "ok",0, 0,0 );
      return KErrEof;
    }
    
    string tmp;
    aIn >> tmp;
    if ( tmp == "#" ) {
      aIn.ignore(4096, '\n');
      
    } else if ( tmp == this->GetLiteral(EClosingTag) ) {
      break;
      
    } else {
      if ( tmp.at( tmp.length()-1 ) == ':' ) {
        int ret = this->ReadParameter(aIn, tmp);
        if ( ret != KErrNone ) {
          return ret;
        }
      } else {
        alert( "Unknown token", tmp.c_str(), 0, "ok",0, 0,0 );
        return KErrRead;
      }
    }
  }
  return KErrNone;
}

int SpawnItem::ReadParameter( istream& aIn, const string& aParameter )
{      
  string tmp = "";
  if ( aParameter == "pos:" ) {
    string posX, posY;
    aIn >> posX >> posY;
    iParameters.addParameter( KParamNamePosX, posX );
    iParameters.addParameter( KParamNamePosY, posY );
    
  } else if ( aParameter == "t:" ) {
    string objSubType;
    aIn >> objSubType;
    iParameters.addParameter( KParamNameSubCategory, objSubType );
    
  } else if ( aParameter == "offset:" ) {
    string offsetX, offsetY;
    aIn >> offsetX >> offsetY;
    iParameters.addParameter( KParamNameOffsetX, offsetX );
    iParameters.addParameter( KParamNameOffsetY, offsetY );
    
  } else if ( aParameter == "id:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameIdCode, tmp );
    
  } else if ( aParameter == "angle:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameAngle, tmp );
    
  } else if ( aParameter == "flags:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameFlags, tmp );
    
  } else if ( aParameter == "route:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameRouteFile, tmp );
    
  } else if ( aParameter == "uid:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameUid, tmp );
    
  } else if ( aParameter == "delay:" ) {
    aIn >> tmp;
    iParameters.addParameter( KParamNameDelay, tmp );
    
  } else {
    aIn >> tmp;
    iParameters.addParameter( aParameter, tmp );
  }
  return KErrNone;
}



///
/// SpawnEvent - class
/// ==================

/** Constructor.
 */
SpawnEvent::SpawnEvent() :
  DynamicEvent()
{
}



/** Destructor.
 */
SpawnEvent::~SpawnEvent()
{
  std::list< SpawnItem* >::iterator iter = m_spawnObjects.begin();
  while ( iter != m_spawnObjects.end() ) {
    delete (*iter);
    iter++;
  }
  m_spawnObjects.clear();
}



/** Reads the data of this event from file.
 */
int SpawnEvent::readData( istream& fin )
{
  LOG_MESSAGE( "Reading Spawn-event" );
  string tmp;

  while ( true ) {
    if ( fin.eof() ) {
      break;
    }
    
    fin >> tmp;
    if ( tmp == "</spawn>" ) {
      return 0;

    } else if ( tmp == "sound:" ) {
      // Read the sound-data.
      fin >> DynamicEvent::m_sound;
      
    } else if ( tmp == "delay:" ) {
      fin >> DynamicEvent::m_counter;
      
    } else if ( tmp == "#" ) {
      fin.ignore( 4098, '\n' );
      
    } else if ( tmp == "difficulty:" ) {
      fin >> tmp;
      DynamicEvent::m_difficulty = 0;
      int bits = 0;
      if ( tmp.find("e",0) != string::npos ) {
        bits |= KBitmaskEasy;
      }
      if ( tmp.find("n",0) != string::npos ) {
        bits |= KBitmaskNormal;
      }
      if ( tmp.find("h",0) != string::npos ) {
        bits |= KBitmaskHard;
      }

      m_difficulty = bits;

    } else {
      LOG_MESSAGE( tmp.c_str() );
      SpawnItem* item = SpawnItemFactory::CreateSpawnItem( tmp );
      if ( item == 0 ) {
        alert( "Parse error:", tmp.c_str(), 0, "ok",0, 0,0 );
        return -1;
      }
      if ( item->Read(fin) != KErrNone ) {
        delete item;
        return -1;
      }
      m_spawnObjects.push_back( item );   
    } 
  }
  return -1;
}



/** Updates this event
 */
void SpawnEvent::update()
{
  if ( DynamicEvent::m_status == DynamicEvent::WAITING ) {
    return;
    
  } else if ( DynamicEvent::m_status == DynamicEvent::ACTIVE ) {
    DynamicEvent::m_counter -= 1;
    if ( DynamicEvent::m_counter < 0 ) {
      this->status( DynamicEvent::COMPLETE );
      this->spawnObjects();
      if ( DynamicEvent::m_sound != -1 ) {
        Sound::playSample( DynamicEvent::m_sound, false );
      }
    }
  }
}


/** Activates this event
 */
void SpawnEvent::activate()
{
  if ( this->isOkToActivate() ) {
    DynamicEvent::m_status = ACTIVE;
  } else {
    DynamicEvent::m_status = COMPLETE;
  }
}

/** Spawns the new objects
 */
void SpawnEvent::spawnObjects()
{
  std::list< SpawnItem* >::iterator iter = m_spawnObjects.begin();
  
  while ( iter != m_spawnObjects.end() ) {
    SpawnItem* item = *iter;
    if ( item != 0 ) {
      item->Launch();
    }    
    iter++;
  }
}

} // end of namespace
